<?php
require 'config.php';

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $email    = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $ref_code = trim($_POST['referral_code'] ?? '');

    // Basic validation
    if (empty($username) || empty($email) || empty($password)) {
        $errors[] = "All fields are required.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Invalid email format.";
    } elseif (strlen($password) < 6) {
        $errors[] = "Password must be at least 6 characters.";
    }

    // Check if username or email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
    $stmt->execute([$username, $email]);
    if ($stmt->fetch()) {
        $errors[] = "Username or email already taken.";
    }

    if (empty($errors)) {
        $hashed = password_hash($password, PASSWORD_DEFAULT);
        $my_ref_code = generateReferralCode();

        $referred_by = null;
        $referrer_id = null;

        if (!empty($ref_code)) {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ?");
            $stmt->execute([$ref_code]);
            $referrer = $stmt->fetch();
            if ($referrer) {
                $referred_by = $ref_code;
                $referrer_id = $referrer['id'];
            }
        }

        // Insert new user
        $stmt = $pdo->prepare("
            INSERT INTO users (username, email, password, referral_code, referred_by)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$username, $email, $hashed, $my_ref_code, $referred_by]);

        // Get the ID of the newly created user
        $new_user_id = $pdo->lastInsertId();

        // ────────────────────────────────────────────────
        // Give referral reward to referrer (0.50 $)
        // ────────────────────────────────────────────────
        if ($referrer_id && $referrer_id != $new_user_id) {  // prevent self-referral
            $stmt = $pdo->prepare("
                UPDATE users 
                SET 
                    balance = balance + 0.50,
                    lifetime_earned = lifetime_earned + 0.50,
                    referral_count = referral_count + 1,
                    monthly_referral_count = monthly_referral_count + 1
                WHERE id = ?
            ");
            $stmt->execute([$referrer_id]);
        }

        $success = "Registration successful! You can now log in.<br>Your referral code: <strong>$my_ref_code</strong>";
    }
}
?>
<!DOCTYPE html>
<html lang="en" class="bg-gray-100">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Rewards App</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="min-h-screen flex items-center justify-center bg-gradient-to-br from-teal-500 to-cyan-600 p-4">

    <div class="w-full max-w-md bg-white rounded-2xl shadow-xl p-8">
        <h1 class="text-3xl font-bold text-center text-teal-700 mb-6">Create Account</h1>

        <?php if ($success): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded">
                <?= $success ?>
            </div>
        <?php endif; ?>

        <?php if ($errors): ?>
            <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded">
                <ul class="list-disc pl-5">
                    <?php foreach ($errors as $err): ?>
                        <li><?= htmlspecialchars($err) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="POST" class="space-y-5">
            <div>
                <label class="block text-gray-700 font-medium">Username</label>
                <input type="text" name="username" required class="mt-1 w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500">
            </div>
            <div>
                <label class="block text-gray-700 font-medium">Email (Gmail or other)</label>
                <input type="email" name="email" required class="mt-1 w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500">
            </div>
            <div>
                <label class="block text-gray-700 font-medium">Password</label>
                <input type="password" name="password" required class="mt-1 w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500">
            </div>
            <div>
                <label class="block text-gray-700 font-medium">Referral Code (optional)</label>
                <input type="text" name="referral_code" class="mt-1 w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500">
            </div>
            <button type="submit" class="w-full bg-teal-600 text-white py-3 rounded-lg font-semibold hover:bg-teal-700 transition">
                Register
            </button>
        </form>

        <p class="text-center mt-6 text-gray-600">
            Already have an account? <a href="login.php" class="text-teal-600 hover:underline">Login</a>
        </p>
    </div>

</body>
</html>