<?php
require 'config.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Handle screenshot submission
$message = '';
$message_type = 'info';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['task_id']) && isset($_FILES['screenshot'])) {
    $task_id = (int)$_POST['task_id'];

    try {
        // Check if task exists and is active
        $stmt = $pdo->prepare("SELECT id, reward_amount, total_slots, used_slots FROM tasks WHERE id = ? AND is_active = 1");
        $stmt->execute([$task_id]);
        $task = $stmt->fetch();

        if (!$task) {
            $message = "Task not found or no longer available.";
            $message_type = 'error';
        } elseif ($task['used_slots'] >= $task['total_slots']) {
            $message = "All slots for this task are taken.";
            $message_type = 'error';
        } else {
            // Check if user already submitted
            $check = $pdo->prepare("SELECT id FROM task_submissions WHERE user_id = ? AND task_id = ?");
            $check->execute([$user_id, $task_id]);
            if ($check->fetch()) {
                $message = "You have already submitted for this task.";
                $message_type = 'warning';
            } else {
                // Handle file upload
                $file = $_FILES['screenshot'];
                $allowed = ['jpg', 'jpeg', 'png'];
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

                if (!in_array($ext, $allowed) || $file['size'] > 5*1024*1024) { // 5MB limit
                    $message = "Invalid file. Use JPG/PNG, max 5MB.";
                    $message_type = 'error';
                } else {
                    $filename = 'screenshot_' . $user_id . '_' . time() . '.' . $ext;
                    $upload_path = 'uploads/tasks/' . $filename;

                    if (!is_dir('uploads/tasks')) {
                        mkdir('uploads/tasks', 0755, true);
                    }

                    if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                        // Record submission
                        $pdo->prepare("
                            INSERT INTO task_submissions 
                            (user_id, task_id, screenshot_url) 
                            VALUES (?, ?, ?)
                        ")->execute([$user_id, $task_id, $upload_path]);

                        // Increment used slots
                        $pdo->prepare("UPDATE tasks SET used_slots = used_slots + 1 WHERE id = ?")
                            ->execute([$task_id]);

                        $message = "Screenshot submitted successfully! Waiting for admin approval.";
                        $message_type = 'success';
                    } else {
                        $message = "Failed to upload screenshot.";
                        $message_type = 'error';
                    }
                }
            }
        }
    } catch (Exception $e) {
        $message = "Error: " . $e->getMessage();
        $message_type = 'error';
    }
}

// Load available tasks
$tasks = $pdo->query("
    SELECT id, title, description, platform, link, reward_amount, total_slots, used_slots
    FROM tasks 
    WHERE is_active = 1 AND used_slots < total_slots
    ORDER BY reward_amount DESC, id DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Get user's submissions status
$submissions = $pdo->prepare("
    SELECT task_id, status 
    FROM task_submissions 
    WHERE user_id = ?
");
$submissions->execute([$user_id]);
$user_submissions = $submissions->fetchAll(PDO::FETCH_KEY_PAIR); // task_id => status
?>

<!DOCTYPE html>
<html lang="en" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tasks - Rewards App</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-teal-500 min-h-screen pb-24 text-gray-800">

    <!-- Header -->
    <header class="bg-teal-600 text-white p-4 flex items-center justify-between sticky top-0 z-20 shadow-md">
        <a href="dashboard.php" class="text-3xl">←</a>
        <h1 class="text-xl font-bold">Tasks & Earn</h1>
        <div class="w-8"></div>
    </header>

    <main class="p-4 max-w-md mx-auto space-y-6">

        <?php if ($message): ?>
            <div class="p-4 rounded-xl <?php
                if ($message_type === 'success') echo 'bg-green-100 text-green-800 border-l-4 border-green-500';
                elseif ($message_type === 'error') echo 'bg-red-100 text-red-800 border-l-4 border-red-500';
                else echo 'bg-yellow-100 text-yellow-800 border-l-4 border-yellow-500';
            ?>">
                <?= $message ?>
            </div>
        <?php endif; ?>

        <div class="bg-white rounded-2xl shadow-xl p-6">
            <h2 class="text-2xl font-bold text-teal-700 mb-5 text-center">Available Tasks</h2>

            <?php if (empty($tasks)): ?>
                <p class="text-center text-gray-600 py-10">
                    No tasks available right now.<br>
                    Check back later!
                </p>
            <?php else: ?>
                <div class="space-y-5">
                    <?php foreach ($tasks as $task): 
                        $status = $user_submissions[$task['id']] ?? null;
                        $slots_left = $task['total_slots'] - $task['used_slots'];
                    ?>
                        <div class="border border-gray-200 rounded-xl overflow-hidden bg-gray-50">
                            <div class="p-5">
                                <div class="flex justify-between items-start mb-2">
                                    <h3 class="font-bold text-lg flex-1">
                                        <?= htmlspecialchars($task['title']) ?>
                                    </h3>
                                    <span class="inline-block px-3 py-1 text-xs font-semibold rounded-full 
                                        <?php
                                            if ($task['platform'] === 'WhatsApp') echo 'bg-green-100 text-green-800';
                                            elseif ($task['platform'] === 'Telegram') echo 'bg-blue-100 text-blue-800';
                                            elseif ($task['platform'] === 'YouTube') echo 'bg-red-100 text-red-800';
                                            elseif ($task['platform'] === 'Facebook') echo 'bg-indigo-100 text-indigo-800';
                                            else echo 'bg-gray-100 text-gray-800';
                                        ?>">
                                        <?= htmlspecialchars($task['platform']) ?>
                                    </span>
                                </div>

                                <p class="text-gray-600 text-sm mb-3">
                                    <?= nl2br(htmlspecialchars($task['description'])) ?>
                                </p>

                                <div class="flex justify-between items-center mb-3 text-sm">
                                    <span class="text-gray-700">
                                        Slots: <strong><?= $task['used_slots'] ?>/<?= $task['total_slots'] ?></strong>
                                        (<?= $slots_left ?> left)
                                    </span>
                                    <span class="text-green-600 font-bold text-lg">
                                        +$<?= number_format($task['reward_amount'], 2) ?>
                                    </span>
                                </div>

                                <a href="<?= htmlspecialchars($task['link']) ?>" target="_blank" rel="noopener noreferrer"
                                   class="block w-full bg-blue-600 text-white text-center py-3 rounded-lg font-medium hover:bg-blue-700 transition mb-4">
                                    Visit Link
                                </a>

                                <?php if ($status === 'pending'): ?>
                                    <p class="text-center text-yellow-600 font-medium py-2">
                                        Your submission is pending approval
                                    </p>
                                <?php elseif ($status === 'approved'): ?>
                                    <p class="text-center text-green-600 font-medium py-2">
                                        ✓ Approved - Reward added
                                    </p>
                                <?php elseif ($status === 'rejected'): ?>
                                    <p class="text-center text-red-600 font-medium py-2">
                                        ✗ Rejected - Please try again
                                    </p>
                                <?php else: ?>
                                    <!-- Show submit form if not submitted -->
                                    <form method="POST" enctype="multipart/form-data" class="space-y-3">
                                        <input type="hidden" name="task_id" value="<?= $task['id'] ?>">

                                        <div>
                                            <label class="block text-gray-700 font-medium mb-1">
                                                Upload Screenshot (proof)
                                            </label>
                                            <input type="file" name="screenshot" accept="image/*" required
                                                   class="w-full p-2 border border-gray-300 rounded-lg">
                                        </div>

                                        <button type="submit"
                                                class="w-full bg-teal-600 text-white py-3 rounded-lg font-semibold hover:bg-teal-700 transition">
                                            Submit Task & Claim Reward
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

    </main>

    <!-- Bottom Nav -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 flex justify-around py-3 shadow-2xl z-20">
        <a href="dashboard.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🏠</span>
            <span class="text-xs">Home</span>
        </a>
        <a href="shop.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🛒</span>
            <span class="text-xs">Shop</span>
        </a>
        <a href="answer.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">❓</span>
            <span class="text-xs">Answer</span>
        </a>
        <a href="more.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">⋯</span>
            <span class="text-xs">More</span>
        </a>
    </nav>

</body>
</html>