<?php
require 'config.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch user data
$stmt = $pdo->prepare("
    SELECT username, email, balance, lifetime_earned, referral_code 
    FROM users 
    WHERE id = ?
");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

// Fetch saved withdrawal methods
$wd_stmt = $pdo->prepare("
    SELECT paypal_email, usdt_address 
    FROM withdrawal_methods 
    WHERE user_id = ?
");
$wd_stmt->execute([$user_id]);
$withdrawal = $wd_stmt->fetch(PDO::FETCH_ASSOC) ?: [
    'paypal_email'  => '',
    'usdt_address'  => ''
];

// Handle form submission (save withdrawal info)
$message = '';
$message_type = 'info';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $paypal = trim($_POST['paypal_email'] ?? '');
    $usdt   = trim($_POST['usdt_address'] ?? '');

    // Basic validation
    $errors = [];
    if (!empty($paypal) && !filter_var($paypal, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Invalid PayPal email format.";
    }
    if (!empty($usdt) && strlen($usdt) < 30) {   // rough check
        $errors[] = "USDT address looks too short.";
    }

    if (empty($errors)) {
        // Upsert (insert or update)
        $pdo->prepare("
            INSERT INTO withdrawal_methods 
            (user_id, paypal_email, usdt_address) 
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                paypal_email = VALUES(paypal_email),
                usdt_address = VALUES(usdt_address),
                updated_at = NOW()
        ")->execute([$user_id, $paypal, $usdt]);

        $message = "Withdrawal details saved successfully!";
        $message_type = 'success';

        // Refresh withdrawal data
        $withdrawal['paypal_email'] = $paypal;
        $withdrawal['usdt_address'] = $usdt;
    } else {
        $message = implode("<br>", $errors);
        $message_type = 'error';
    }
}

// Build referral link (change domain to your real one)
$base_url = "https://yourdomain.com/register.php";   // ← CHANGE THIS
$referral_link = $base_url . "?ref=" . urlencode($user['referral_code']);
?>

<!DOCTYPE html>
<html lang="en" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>More - Rewards App</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-teal-500 min-h-screen pb-24 text-gray-800">

    <!-- Header -->
    <header class="bg-teal-600 text-white p-4 flex items-center justify-between sticky top-0 z-20 shadow-md">
        <a href="dashboard.php" class="text-3xl font-bold">←</a>
        <h1 class="text-xl font-bold">More</h1>
        <div class="w-8"></div>
    </header>

    <main class="p-4 max-w-md mx-auto space-y-6">

        <!-- User Information -->
        <div class="bg-white rounded-2xl shadow-xl p-6">
            <h2 class="text-2xl font-bold text-teal-700 mb-5 text-center">Your Profile</h2>

            <div class="space-y-4">
                <div class="flex justify-between">
                    <span class="text-gray-600">Username:</span>
                    <span class="font-semibold"><?= htmlspecialchars($user['username']) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Email:</span>
                    <span class="font-semibold"><?= htmlspecialchars($user['email']) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Current Balance:</span>
                    <span class="font-bold text-green-600">$<?= number_format($user['balance'], 2) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Lifetime Earned:</span>
                    <span class="font-semibold">$<?= number_format($user['lifetime_earned'], 2) ?></span>
                </div>
            </div>
        </div>

        <!-- Referral Info -->
        <div class="bg-white rounded-2xl shadow-xl p-6">
            <h2 class="text-2xl font-bold text-teal-700 mb-4 text-center">Refer & Earn</h2>
            <p class="text-center text-gray-600 mb-4">
                Invite friends and earn <strong>$50</strong> per successful referral!
            </p>

            <div class="space-y-4">
                <div>
                    <label class="block text-sm text-gray-600 mb-1">Your Referral Code</label>
                    <div class="flex items-center bg-gray-100 rounded-lg p-3">
                        <span class="font-mono font-bold text-xl flex-1"><?= htmlspecialchars($user['referral_code']) ?></span>
                        <button onclick="navigator.clipboard.writeText('<?= htmlspecialchars($user['referral_code']) ?>'); alert('Referral code copied!')" 
                                class="ml-2 text-teal-600 hover:text-teal-800">
                            📋
                        </button>
                    </div>
                </div>

                <div>
                    <label class="block text-sm text-gray-600 mb-1">Referral Link</label>
                    <div class="flex items-center bg-gray-100 rounded-lg p-3 break-all">
                        <span class="flex-1 text-sm"><?= htmlspecialchars($referral_link) ?></span>
                        <button onclick="navigator.clipboard.writeText('<?= htmlspecialchars($referral_link) ?>'); alert('Link copied!')" 
                                class="ml-2 text-teal-600 hover:text-teal-800">
                            📋
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Withdrawal Methods -->
        <div class="bg-white rounded-2xl shadow-xl p-6">
            <h2 class="text-2xl font-bold text-teal-700 mb-5 text-center">Withdrawal Settings</h2>
            <p class="text-center text-gray-600 mb-6">
                Add your PayPal or USDT address to enable cash out.
            </p>

            <?php if ($message): ?>
            <div class="p-4 rounded-xl mb-6 <?php
                echo $message_type === 'success' ? 'bg-green-100 text-green-800 border-l-4 border-green-500' :
                     'bg-red-100 text-red-800 border-l-4 border-red-500';
            ?>">
                <?= $message ?>
            </div>
            <?php endif; ?>

            <form method="POST" class="space-y-5">
                <div>
                    <label class="block text-gray-700 font-medium mb-1">PayPal Email</label>
                    <input type="email" name="paypal_email" value="<?= htmlspecialchars($withdrawal['paypal_email']) ?>"
                           class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500"
                           placeholder="yourpaypal@gmail.com">
                </div>

                <div>
                    <label class="block text-gray-700 font-medium mb-1">USDT Address (TRC20 / ERC20 / BEP20)</label>
                    <input type="text" name="usdt_address" value="<?= htmlspecialchars($withdrawal['usdt_address']) ?>"
                           class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500"
                           placeholder="T... or 0x...">
                </div>

                <button type="submit" 
                        class="w-full bg-teal-600 text-white py-4 rounded-xl font-bold hover:bg-teal-700 transition active:scale-95">
                    Save Withdrawal Details
                </button>
            </form>

            <p class="text-center text-sm text-gray-500 mt-6">
                You can update these anytime. Cash out requests will use the saved info.
            </p>
        </div>

    </main>

    <!-- Bottom Navigation -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 flex justify-around py-3 shadow-2xl z-20">
        <a href="dashboard.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🏠</span>
            <span class="text-xs">Home</span>
        </a>
        <a href="shop.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🛒</span>
            <span class="text-xs">Shop</span>
        </a>
        <a href="answer.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">❓</span>
            <span class="text-xs">Answer</span>
        </a>
        <a href="more.php" class="flex flex-col items-center text-teal-600">
            <span class="text-2xl">⋯</span>
            <span class="text-xs font-medium">More</span>
        </a>
    </nav>

</body>
</html>