<?php
require 'config.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// ────────────────────────────────────────────────
//  HANDLE ANSWER SUBMISSION (POST request)
// ────────────────────────────────────────────────
$message = '';
$message_type = 'info';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['question_id'], $_POST['answer'])) {
    $qid = (int)$_POST['question_id'];
    $user_answer = trim($_POST['answer']);

    // Get question data
    $stmt = $pdo->prepare("
        SELECT id, correct_answer, reward_amount, is_active 
        FROM questions 
        WHERE id = ? AND is_active = 1
    ");
    $stmt->execute([$qid]);
    $question = $stmt->fetch();

    if (!$question) {
        $message = "Question not found or no longer available.";
        $message_type = 'error';
    } else {
        // Check if user already answered this correctly
        $check = $pdo->prepare("
            SELECT is_correct, earned_amount 
            FROM user_answers 
            WHERE user_id = ? AND question_id = ?
        ");
        $check->execute([$user_id, $qid]);
        $existing = $check->fetch();

        if ($existing && $existing['is_correct'] == 1) {
            $message = "You already answered this question correctly.";
            $message_type = 'warning';
        } else {
            $is_correct = (strtolower($user_answer) === strtolower($question['correct_answer']));
            $earned = $is_correct ? $question['reward_amount'] : 0.00;

            // Record answer
            $pdo->prepare("
                INSERT INTO user_answers 
                (user_id, question_id, user_answer, is_correct, earned_amount)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    user_answer = VALUES(user_answer),
                    is_correct = VALUES(is_correct),
                    earned_amount = VALUES(earned_amount),
                    answered_at = NOW()
            ")->execute([$user_id, $qid, $user_answer, $is_correct ? 1 : 0, $earned]);

            if ($is_correct && $earned > 0) {
                // Add to user balance
                $pdo->prepare("
                    UPDATE users 
                    SET balance = balance + ?, 
                        lifetime_earned = lifetime_earned + ?
                    WHERE id = ?
                ")->execute([$earned, $earned, $user_id]);

                $message = "Correct! You earned <strong>$$earned</strong>!";
                $message_type = 'success';
            } else {
                $message = "Incorrect answer. Try again later.";
                $message_type = 'error';
            }
        }
    }
}

// ────────────────────────────────────────────────
//  GET ALL ACTIVE QUESTIONS + user's progress
// ────────────────────────────────────────────────
$questions = $pdo->query("
    SELECT q.id, q.question_text, q.reward_amount, q.difficulty,
           COALESCE(ua.is_correct, 0) AS already_correct
    FROM questions q
    LEFT JOIN user_answers ua ON ua.question_id = q.id AND ua.user_id = $user_id
    WHERE q.is_active = 1
    ORDER BY q.reward_amount DESC, q.id ASC
")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en" class="scroll-smooth">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Answer & Earn - Rewards App</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-teal-500 min-h-screen pb-24 text-gray-800">

    <!-- Header -->
    <header class="bg-teal-600 text-white p-4 flex items-center justify-between sticky top-0 z-20 shadow-md">
        <a href="dashboard.php" class="text-3xl">←</a>
        <h1 class="text-xl font-bold">Answer & Earn</h1>
        <div class="w-8"></div>
    </header>

    <main class="p-4 max-w-md mx-auto space-y-6">

        <?php if ($message): ?>
        <div class="p-4 rounded-xl <?php
            echo $message_type === 'success' ? 'bg-green-100 text-green-800 border-l-4 border-green-500' :
                 $message_type === 'error'   ? 'bg-red-100 text-red-800 border-l-4 border-red-500' :
                 'bg-yellow-100 text-yellow-800 border-l-4 border-yellow-500';
        ?>">
            <?= $message ?>
        </div>
        <?php endif; ?>

        <div class="bg-white rounded-2xl shadow-xl p-6">
            <h2 class="text-2xl font-bold text-teal-700 mb-5 text-center">
                Available Questions
            </h2>

            <?php if (empty($questions)): ?>
                <p class="text-center text-gray-600 py-10">
                    No questions available right now.<br>
                    Check back later!
                </p>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($questions as $q): ?>
                    **Summary:**

                            <div class="flex-1">
                                <p class="font-medium text-gray-800">
                                    <?= htmlspecialchars($q['question_text']) ?>
                                </p>
                                <p class="text-sm text-green-600 font-semibold mt-1">
                                    Reward: $<?= number_format($q['reward_amount'], 2) ?>
                                    <?php if ($q['already_correct']): ?>
                                        <span class="text-xs text-gray-500">(already earned)</span>
                                    <?php endif; ?>
                                </p>
                            </div>
                            <span class="text-teal-600 group-open:rotate-180 transition-transform">
                                ▼
                            </span>
                        

                        <?php if (!$q['already_correct']): ?>
                        <div class="p-4 pt-0 bg-white border-t">
                            <form method="POST" class="space-y-4">
                                <input type="hidden" name="question_id" value="<?= $q['id'] ?>">

                                <input type="text" name="answer"
                                       placeholder="Type your answer here..."
                                       class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-teal-500"
                                       required autocomplete="off">

                                <button type="submit"
                                        class="w-full bg-teal-600 text-white py-3 rounded-lg font-semibold hover:bg-teal-700 transition">
                                    Submit Answer
                                </button>
                            </form>
                        </div>
                        <?php else: ?>
                        <div class="p-4 pt-0 text-center text-green-600">
                            ✓ You already answered correctly and earned the reward.
                        </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

    </main>

    <!-- Bottom Nav (same as other pages) -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 flex justify-around py-3 shadow-2xl z-20">
        <a href="dashboard.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🏠</span>
            <span class="text-xs">Home</span>
        </a>
        <a href="shop.php" class="flex flex-col items-center text-gray-600 hover:text-teal-600">
            <span class="text-2xl">🛒</span>
            <span class="text-xs">Shop</span>
        </a>
        <a href="answer.php" class="flex flex-col items-center text-teal-600">
            <span class="text-2xl">❓</span>
            <span class="text-xs font-medium">Answer</span>
        </a>
        <a href="logout.php" class="flex flex-col items-center text-gray-600 hover:text-red-600">
            <span class="text-2xl">⋯</span>
            <span class="text-xs">More</span>
        </a>
    </nav>

</body>
</html>